/*------------------------------------------------------------------------
  **** BEGIN LICENSE BLOCK ****
  Version: MPL 1.1/GPL 2.0/LGPL 2.1
  Copyright(c) 2004-2010 Created by Eric Fredericksen (www.pttpsystems.com)
  All Rights Reserved.

  This program are subject to the Mozilla Public License Version
  1.1 (the "License"); you may not use this file except in compliance with
  the License. You may obtain a copy of the License at
  http://www.mozilla.org/MPL/

  This program is free software. Software distributed under the License
  is distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND,
  either express or implied. See the License for the specific language
  governing rights and limitations under the License.

  Alternatively, the contents of this file may be used under the terms of
  either the GNU General Public License Version 2 or later (the "GPL"), or
  the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
  in which case the provisions of the GPL or the LGPL are applicable instead
  of those above. If you wish to allow use of your version of this file only
  under the terms of either the GPL or the LGPL, and not to allow others to
  use your version of this file under the terms of the MPL, indicate your
  decision by deleting the provisions above and replace them with the notice
  and other provisions required by the GPL or the LGPL. If you do not delete
  the provisions above, a recipient may use your version of this file under
  the terms of any one of the MPL, the GPL or the LGPL.

  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc., 59 Temple
  Place, Suite 330, Boston, MA 02111-1307 USA
  **** END LICENSE BLOCK ****
------------------------------------------------------------------------*/

#include "stdafx.h"

#include "AnagramServer.h"

//=====================================================================
// one global dictionary for memory conservation 
//=====================================================================
CAnagramDictionary g_oAnagramDictionary(_T("wordlist.txt"));

//=====================================================================
// the worker threads only use one dictionary in this class-static variable
//=====================================================================
CAnagramDictionary * CAnagramThread::m_poAnagramDictionary = & g_oAnagramDictionary;


//==================================================
// read the string into a buffer, null terminate, check for errors
//==================================================
bool CAnagramThread::ReadRequestString(void)
{
	//zero the buffer for safety.
	RtlZeroMemory(&m_facRequestStringBuffer, sizeof(m_facRequestStringBuffer) );

	// read as many bytes as we can up to max length
	int iResult = 
		recv(m_oMySocket, 
		&m_facRequestStringBuffer[0], 
		sizeof(m_facRequestStringBuffer), 
		0 // no (mis)behaving peek or oob
		);
	// ensure null termination after read
	m_facRequestStringBuffer[MAX_REQUEST_READ] = 0;


	if( 0 == iResult )
	{
		// socket closed without input or wait for return? save error message
		fprintf(stderr, _T("[%u]Socket already closed on recv()\n"),m_dwThreadID );
		return(false);
	}
	else if( SOCKET_ERROR  == iResult )
	{
		// save error message on socket error
		fprintf(stderr, _T("[%u]Socket error on recv()\n"),m_dwThreadID );
		return(false);
	}
	else if(WSAETIMEDOUT == iResult )
	{
		// I assume that a timeout of 30 seconds is sufficient to say the caller is misbehaving
		// save error message
		fprintf(stderr, _T("[%u]Socket timeout on recv()\n"),m_dwThreadID );
		return(false);
	}
	fprintf(stderr, _T("<==[%u]ReadReq[%s]\n"),m_dwThreadID, m_facRequestStringBuffer);

	return(true);
}

//==================================================
//
//==================================================
void CAnagramThread::ProcessRequestString(void)
{
	CAnagramSignature oSignature(m_facRequestStringBuffer);

	// us our static member variable to look stuff up in a thread safe way
	string szResponse = m_poAnagramDictionary->ThreadSafeLookup(oSignature);
	int iResult = send(m_oMySocket, szResponse.c_str(), (int)szResponse.size(), 0);

	if( SOCKET_ERROR == iResult )
	{
		// record error
		fprintf(stderr, _T("[%u]Socket error on send()\n"), m_dwThreadID );
	}
	else if( ((size_t)iResult) < szResponse.size() )
	{
		// record error on partial send
		fprintf(stderr, _T("[%u]Not all bytes were sent on send()\n"), m_dwThreadID );
	}
	else if(WSAETIMEDOUT == iResult )
	{
		// I assume that a timeout of 30 seconds is sufficient to say the caller is misbehaving
		// save error message
		fprintf(stderr, _T("[%u]Socket timeout on send()\n"), m_dwThreadID );
	}
	else
	{
		// print out the response :)
		fprintf(stderr,"==>[%u]Processed[%s]=><%s>\n", m_dwThreadID, &m_facRequestStringBuffer[0], szResponse.c_str());
	}

}

//==================================================
// do all of our work-load setup stuff here
//==================================================
void CAnagramThread::AssignThread(SOCKET _oMySocket)
{

	// save the socket
	m_oMySocket = _oMySocket;

	// make sure the timeouts are what we want
	DWORD dwTimeout_ms = DEFAULT_SOCKET_TIMEOUT_ms;
	int iResult =
		setsockopt(m_oMySocket, SOL_SOCKET, SO_RCVTIMEO, (const char*)&dwTimeout_ms, sizeof(dwTimeout_ms) );
	if( SOCKET_ERROR  == iResult )
	{
		// send warning message
	}
	iResult =
		setsockopt(m_oMySocket, SOL_SOCKET, SO_SNDTIMEO, (const char*)&dwTimeout_ms, sizeof(dwTimeout_ms) );
	if( SOCKET_ERROR  == iResult )
	{
		// send warning message
	}

	// to protect against race conditions, use the non waiting, non running state
	m_dwThreadState = ASSIGNED;

	// release the hounds -- signal thread to start
	ReleaseThread(); 
}

//==================================================
// override this for doing our work
//==================================================
DWORD CAnagramThread::VirtualCallbackFunction(void)
{
	DWORD dwResult = FALSE;
	if( ReadRequestString() )
	{
		ProcessRequestString();
		dwResult = TRUE;
	}

	// close down the socket
	shutdown(m_oMySocket, SD_BOTH);
	// wait for a bit in case the socket is constipated
	Sleep(DEFAULT_SHUTDOWN_WAIT_ms);
	// close the socket so it can be recycled
	closesocket(m_oMySocket);

	return(dwResult);
}


//==================================================
//
//==================================================
bool CAnagramServer::InitSuccess(void)
{
	if( m_bInitSuccess ) return(true);

	if( !CServer::InitSuccess() ) return(false);

	for(long K=0; K<DEFAULT_WORKTHREAD_COUNT; K++)
	{
		if( !m_faoWorkThreads[K].InitSuccess() )
		{
			// give error message
			fprintf(stderr, _T("Workerthread %d did not initialize\n"), K);
			return(false);
		}
	}
	m_bInitSuccess = true;
	return(m_bInitSuccess);
}

//==================================================
//
//==================================================
CAnagramServer::CAnagramServer(void)
{
	// nothing special to do here?
}

//==================================================
//
//==================================================
CAnagramServer::~CAnagramServer()
{
	// nothing special to do here?
}

//==================================================
// Find a thread to do the work and assign the socket
//==================================================
void CAnagramServer::DispatchConnection(SOCKET _sWorkSocket)
{
	if( !m_bInitSuccess ) return;

	bool bHandedOff = false;
	do {
		// find a ready worker thread. wait around if all our threads are busy
		for(long K=0; K<DEFAULT_WORKTHREAD_COUNT; K++)
		{
			if( CAnagramThread::WAITING == m_faoWorkThreads[K].GetThreadState() )
			{
				// hand off the socket and start the thread
				m_faoWorkThreads[K].AssignThread(_sWorkSocket);
				bHandedOff = true;
				break;
			}
		}
		// wait before checking again to give threads time to finish their jobs
		// and so we don't busy wait here using up CPU. :)
		if( !bHandedOff ) Sleep(100);
	} while(!bHandedOff);
}

